/*
 * Copyright (c) 1999-2001,2005-2012 Apple Inc. All Rights Reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 *
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 *
 * @APPLE_LICENSE_HEADER_END@
 */

/*
 * SecureTransportPriv.h - Apple-private exported routines
 */

#ifndef	_SECURE_TRANSPORT_PRIV_H_
#define _SECURE_TRANSPORT_PRIV_H_	1

#include <Security/SecureTransport.h>
#include <Security/SecTrust.h>

#ifdef __cplusplus
extern "C" {
#endif

/* The size of of client- and server-generated random numbers in hello messages. */
#define SSL_CLIENT_SRVR_RAND_SIZE		32

/* The size of the pre-master and master secrets. */
#define SSL_RSA_PREMASTER_SECRET_SIZE	48
#define SSL_MASTER_SECRET_SIZE			48

/*
 * For the following three functions, *size is the available
 * buffer size on entry and the actual size of the data returned
 * on return. The above consts are for convenience.
 */
OSStatus SSLInternalMasterSecret(
   SSLContextRef context,
   void *secret,         // mallocd by caller, SSL_MASTER_SECRET_SIZE
   size_t *secretSize);  // in/out

OSStatus SSLInternalServerRandom(
   SSLContextRef context,
   void *randBuf, 			// mallocd by caller, SSL_CLIENT_SRVR_RAND_SIZE
   size_t *randSize);	// in/out

OSStatus SSLInternalClientRandom(
   SSLContextRef context,
   void *randBuf,  		// mallocd by caller, SSL_CLIENT_SRVR_RAND_SIZE
   size_t *randSize);	// in/out

/*
 * Obtain the sizes of the currently negotiated HMAC digest, session
 * key, and session key IV.
 */
OSStatus SSLGetCipherSizes(
	SSLContextRef context,
	size_t *digestSize,
	size_t *symmetricKeySize,
	size_t *ivSize);

OSStatus SSLInternal_PRF(
   SSLContextRef context,
   const void *secret,
   size_t secretLen,
   const void *label,
   size_t labelLen,
   const void *seed,
   size_t seedLen,
   void *out,   		// mallocd by caller, length >= outLen
   size_t outLen);

/*
 * Obtain a SecTrustRef representing peer certificates. Valid anytime,
 * subsequent to a handshake attempt. The returned SecTrustRef is valid
 * only as long as the SSLContextRef is.
 */
OSStatus
SSLGetPeerSecTrust			(SSLContextRef 		context,
							 SecTrustRef		*secTrust);	/* RETURNED */

/*
 * Obtain resumable session info. Can be called anytime subsequent to
 * handshake attempt.
 *
 * if sessionWasResumed is True on return, the session is indeed a
 * resumed session; the sessionID (an opaque blob generated by the
 * server) is returned in *sessionID. The length of the sessionID
 * is returned in *sessionIDLength. Caller must allocate the
 * sessionID buffer; it max size is MAX_SESSION_ID_LENGTH bytes.
 */
#define MAX_SESSION_ID_LENGTH	32

OSStatus
SSLGetResumableSessionInfo	(
	SSLContextRef	context,
	Boolean			*sessionWasResumed,		// RETURNED
	void			*sessionID,				// RETURNED, mallocd by caller
	size_t			*sessionIDLength);		// IN/OUT

/*
 * Getters for SSLSetCertificate() and SSLSetEncryptionCertificate()
 */
OSStatus
SSLGetCertificate (
	SSLContextRef	context,
	CFArrayRef		*certRefs);				// RETURNED, *not* retained

OSStatus
SSLGetEncryptionCertificate (
	SSLContextRef	context,
	CFArrayRef		*certRefs);				// RETURNED, *not* retained

/*
 * Getter for SSLSetClientSideAuthenticate()
 */
OSStatus
SSLGetClientSideAuthenticate (
	SSLContextRef	context,
	SSLAuthenticate	*auth);					// RETURNED

/*
 * Get/set array of trusted leaf certificates.
 *
 * If none have been set previously with SSLSetTrustedLeafCertificates(),
 * then SSLCopyTrustedLeafCertificates() will return NULL with noErr.
 */
OSStatus
SSLSetTrustedLeafCertificates (
	SSLContextRef	context,
	CFArrayRef 		certRefs);

OSStatus
SSLCopyTrustedLeafCertificates (
	SSLContextRef	context,
	CFArrayRef 		*certRefs);				// RETURNED, caller must release

/*
 * Get/set enable of anonymous ciphers. Default is enabled.
 *
 * SSLSetAllowAnonymousCiphers() returns badReqErr if SSLSetEnabledCiphers()
 * has already been called.
 *
 * The enable state set by SSLSetAllowAnonymousCiphers() is ignored if
 * SSLSetEnabledCiphers() is called after SSLSetAllowAnonymousCiphers() is
 * called, i.e., SSLSetEnabledCiphers() overrides SSLSetAllowAnonymousCiphers().
 *
 * NOTE: "Anonymous" ciphers include those ciphers that perform no encryption,
 * as well as ciphers that perform no authentication, since neither are secure.
 */
OSStatus
SSLSetAllowAnonymousCiphers(
	SSLContextRef	context,
	Boolean			enable);

OSStatus
SSLGetAllowAnonymousCiphers(
	SSLContextRef	context,
	Boolean			*enable);

/*
 * Override the default session cache timeout for a cache entry created for
 * the current session.
 */
OSStatus
SSLSetSessionCacheTimeout(
	SSLContextRef context,
	uint32_t timeoutInSeconds);

/*
 * Callback function for EAP-style PAC-based session resumption.
 * This function is called by SecureTransport to obtain the
 * master secret.
 */
typedef void (*SSLInternalMasterSecretFunction)(
	SSLContextRef ctx,
	const void *arg,		/* opaque to SecureTransport; app-specific */
	void *secret,			/* mallocd by caller, SSL_MASTER_SECRET_SIZE */
	size_t *secretLength);  /* in/out */

/*
 * Register a callback for obtaining the master_secret when performing
 * PAC-based session resumption. At the time the callback is called,
 * the following are guaranteed to be valid:
 *
 *  -- serverRandom (via SSLInternalServerRandom())
 *  -- clientRandom (via SSLInternalClientRandom())
 *  -- negotiated protocol version (via SSLGetNegotiatedProtocolVersion())
 *  -- negotiated CipherSuite (via SSLGetNegotiatedCipher())
 *
 * Currently, PAC-based session resumption is only implemented on
 * the client side for Deployment builds.
 *
 * On the client side, this callback occurs if/when the server sends a
 * ChangeCipherSpec message immediately following its ServerHello
 * message (i.e., it's skipped the entire Key Exchange phase of
 * negotiation).
 *
 * On the server side (Development builds only) this callback occurs
 * immediately upon receipt of the Client Hello message, before we send
 * the Server Hello.
 */
OSStatus
SSLInternalSetMasterSecretFunction(
	SSLContextRef ctx,
	SSLInternalMasterSecretFunction mFunc,
	const void *arg);		/* opaque to SecureTransport; app-specific */

/*
 * Provide an opaque SessionTicket for use in PAC-based session
 * resumption. Client side only. The provided ticket is sent in
 * the ClientHello message as a SessionTicket extension.
 * The maximum ticketLength is 2**16-1.
 */
OSStatus SSLInternalSetSessionTicket(
   SSLContextRef ctx,
   const void *ticket,
   size_t ticketLength);

/*
 * Support for specifying and obtaining ECC curves, used with the ECDH-based
 * ciphersuites.
 */

/*
 * These are the named curves from RFC 4492
 * section 5.1.1, with the exception of SSL_Curve_None which means
 * "ECDSA not negotiated".
 */
typedef enum
{
	SSL_Curve_None = -1,

    SSL_Curve_sect163k1 = 1,
    SSL_Curve_sect163r1 = 2,
    SSL_Curve_sect163r2 = 3,
    SSL_Curve_sect193r1 = 4,
    SSL_Curve_sect193r2 = 5,
    SSL_Curve_sect233k1 = 6,
    SSL_Curve_sect233r1 = 7,
    SSL_Curve_sect239k1 = 8,
    SSL_Curve_sect283k1 = 9,
    SSL_Curve_sect283r1 = 10,
    SSL_Curve_sect409k1 = 11,
    SSL_Curve_sect409r1 = 12,
    SSL_Curve_sect571k1 = 13,
    SSL_Curve_sect571r1 = 14,
    SSL_Curve_secp160k1 = 15,
    SSL_Curve_secp160r1 = 16,
    SSL_Curve_secp160r2 = 17,
    SSL_Curve_secp192k1 = 18,
    SSL_Curve_secp192r1 = 19,
    SSL_Curve_secp224k1 = 20,
    SSL_Curve_secp224r1 = 21,
    SSL_Curve_secp256k1 = 22,

    /* These are the ones we actually support */
	SSL_Curve_secp256r1 = 23,
	SSL_Curve_secp384r1 = 24,
	SSL_Curve_secp521r1 = 25
} SSL_ECDSA_NamedCurve;

/*
 * Obtain the SSL_ECDSA_NamedCurve negotiated during a handshake.
 * Returns paramErr if no ECDH-related ciphersuite was negotiated.
 */
extern OSStatus SSLGetNegotiatedCurve(
   SSLContextRef ctx,
   SSL_ECDSA_NamedCurve *namedCurve);    /* RETURNED */

/*
 * Obtain the number of currently enabled SSL_ECDSA_NamedCurves.
 */
extern OSStatus SSLGetNumberOfECDSACurves(
   SSLContextRef ctx,
   unsigned *numCurves);				/* RETURNED */

/*
 * Obtain the ordered list of currently enabled SSL_ECDSA_NamedCurves.
 * Caller allocates returned array and specifies its size (in
 * SSL_ECDSA_NamedCurves) in *numCurves on entry; *numCurves
 * is the actual size of the returned array on successful return.
 */
extern OSStatus SSLGetECDSACurves(
   SSLContextRef ctx,
   SSL_ECDSA_NamedCurve *namedCurves,	/* RETURNED */
   unsigned *numCurves);				/* IN/OUT */

/*
 * Specify ordered list of allowable named curves.
 */
extern OSStatus SSLSetECDSACurves(
   SSLContextRef ctx,
   const SSL_ECDSA_NamedCurve *namedCurves,
   unsigned numCurves);

/*
 * Server-specified client authentication mechanisms.
 */
typedef enum {
	/* doesn't appear on the wire */
	SSLClientAuthNone = -1,
	/* RFC 2246 7.4.6 */
	SSLClientAuth_RSASign = 1,
	SSLClientAuth_DSSSign = 2,
	SSLClientAuth_RSAFixedDH = 3,
	SSLClientAuth_DSS_FixedDH = 4,
	/* RFC 4492 5.5 */
	SSLClientAuth_ECDSASign = 64,
	SSLClientAuth_RSAFixedECDH = 65,
	SSLClientAuth_ECDSAFixedECDH = 66
} SSLClientAuthenticationType;

/* TLS 1.2 Signature Algorithms extension values for hash field. */
typedef enum {
    SSL_HashAlgorithmNone = 0,
    SSL_HashAlgorithmMD5 = 1,
    SSL_HashAlgorithmSHA1 = 2,
    SSL_HashAlgorithmSHA224 = 3,
    SSL_HashAlgorithmSHA256 = 4,
    SSL_HashAlgorithmSHA384 = 5,
    SSL_HashAlgorithmSHA512 = 6
} SSL_HashAlgorithm;

/* TLS 1.2 Signature Algorithms extension values for signature field. */
typedef enum {
    SSL_SignatureAlgorithmAnonymous = 0,
    SSL_SignatureAlgorithmRSA = 1,
    SSL_SignatureAlgorithmDSA = 2,
    SSL_SignatureAlgorithmECDSA = 3
} SSL_SignatureAlgorithm;

typedef struct {
    SSL_HashAlgorithm hash;
    SSL_SignatureAlgorithm signature;
} SSLSignatureAndHashAlgorithm;

/*
 * Obtain the number of client authentication mechanisms specified by
 * the server in its Certificate Request message.
 * Returns paramErr if server hasn't sent a Certificate Request message
 * (i.e., client certificate state is kSSLClientCertNone).
 */
extern OSStatus SSLGetNumberOfClientAuthTypes(
	SSLContextRef ctx,
	unsigned *numTypes);

/*
 * Obtain the client authentication mechanisms specified by
 * the server in its Certificate Request message.
 * Caller allocates returned array and specifies its size (in
 * SSLClientAuthenticationTypes) in *numType on entry; *numTypes
 * is the actual size of the returned array on successful return.
 */
extern OSStatus SSLGetClientAuthTypes(
   SSLContextRef ctx,
   SSLClientAuthenticationType *authTypes,		/* RETURNED */
   unsigned *numTypes);							/* IN/OUT */

/*
 * Obtain the SSLClientAuthenticationType actually performed.
 * Only valid if client certificate state is kSSLClientCertSent
 * or kSSLClientCertRejected; SSLClientAuthNone is returned as
 * the negotiated auth type otherwise.
 */
extern OSStatus SSLGetNegotiatedClientAuthType(
   SSLContextRef ctx,
   SSLClientAuthenticationType *authType);		/* RETURNED */

/* Private SSL session options */
typedef enum {
	/*
	 * This option can be used to enable sending the first byte
	 * of application data in its own SSL record in order to
	 * mitigate a known-IV weakness, a.k.a. the BEAST attack.
	 */
	kSSLSessionOptionSendOneByteRecord = -1
} SSLPrivateSessionOption;

#ifdef __cplusplus
}
#endif

#endif	/* _SECURE_TRANSPORT_PRIV_H_ */
